<?php
/**
 * @package   mod_adsimple_acm
 * @copyright Copyright (C) 2025 AdSimple GmbH. All rights reserved.
 * @license   GNU General Public License version 2, or later
 */
defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;

/**
 * Helper class for the AdSimple ACM module.
 *
 * Provides utility methods for validating ACM IDs, building script URLs,
 * and managing module-related configuration logic.
 *
 * This class is used by the module entry point (mod_adsimple_acm.php)
 * and may also be referenced from templates or other extensions that
 * interact with the AdSimple Consent Manager integration.
 */
class ModAdsimpleAcmHelper
{
/**
 * Validate AdSimple ACM ID using the official API.
 *
 * @param string $acmId
 * @return bool
 */

public const ADSIMPLE_API_URL = 'https://www.adsimple.at/wp-json/as_csc/v1/embed_code?id_key=%s&domain=%s&type=notice,list';

public static function validateAdsimpleId($acmId)
{
	if (empty($acmId)) {
		return false;
	}

	$domain = $_SERVER['SERVER_NAME'] ?? 'localhost';
	$apiUrl = sprintf(
		self::ADSIMPLE_API_URL,
		urlencode(trim($acmId)),
		urlencode($domain)
	);

	try {
		$http = \Joomla\CMS\Http\HttpFactory::getHttp();
		$response = $http->get($apiUrl);

		if ($response->getStatusCode() !== 200) {
			return false;
		}

		$body = json_decode($response->body, true);

		if ( ! is_array($body) || ! isset($body['status'], $body['code']) ) {
			return false;
		}

		$connection_status = trim( $body['status'] ?? '' );
		$connection_code   = $body['code'] ?? 0;
		
		if ( $connection_status == 'invalid_adsimple_id' && $connection_code == 200 ) {
			Factory::getApplication()->enqueueMessage(
				Text::_('MOD_ADSIMPLE_ACM_ERROR_INVALID_ID'), 
				'error'
			);
			return false;
		}

		if ( $connection_status == 'domain_not_exist' && $connection_code == 400)  {
			Factory::getApplication()->enqueueMessage(
				Text::_('MOD_ADSIMPLE_ACM_ERROR_INVALID_DOMAIN'), 
				'error'
			);
			return false;
		}

		if ( $connection_status == 'success' && $connection_code == 200 ) {
			return true;
		}

	} catch (Exception $e) {
		Factory::getApplication()->enqueueMessage(
			Text::_('MOD_ADSIMPLE_ACM_ERROR_FETCH_FAILED'),
			'error'
		);

		Log::add( $e->getMessage(), Log::ERROR, 'adsimple-acm-fetch-failed' );
	}

	return false;
}

/**
 * Retrieve the remote AdSimple Consent Manager JavaScript URL for a given ACM ID.
 * This method contacts AdSimple’s public REST API to fetch the embed code
 * associated with the provided ACM ID and extracts the <script> URL from
 * the returned HTML snippet.
 * If any step fails (invalid ID, bad response, or missing script tag),
 * an empty string is returned.
 */
public static function getScriptUrlFromAdsimple($acmId)
{
	if (empty($acmId)) {
		return '';
	}

	$result = array();

	$domain = $_SERVER['SERVER_NAME'] ?? 'localhost';
	$apiUrl = sprintf(
		self::ADSIMPLE_API_URL,
		urlencode(trim($acmId)),
		urlencode($domain)
	);

	try {
		$http = \Joomla\CMS\Http\HttpFactory::getHttp();
		$response = $http->get($apiUrl);

		if ($response->getStatusCode() !== 200) {
			return '';
		}

		$data = json_decode($response->body, true);

		if ( ! is_array($data) || empty($data['result']) ) {
			return '';
		}

		// Extract *all* script src attributes from the embed HTML
		if (preg_match_all('/<script[^>]+src=["\']([^"\']+)["\'][^>]*>/i', $data['result'], $matches)) {

			// $matches[1] will contain an array of all URLs found
			$scripts = $matches[1];

			// Optional: map them to meaningful keys if needed
			$result = [
				'notice'     => $scripts[0] ?? '',
				'notic_list' => $scripts[1] ?? ''
			];

			return $result; // return both URLs
		}

	} catch (Exception $e) {
		Factory::getApplication()->enqueueMessage(
			Text::_('MOD_ADSIMPLE_ACM_ERROR_FETCH_FAILED'),
			'error'
		);

		Log::add( $e->getMessage(), Log::ERROR, 'adsimple-acm-fetch-failed' );
	}

	return '';
}

/**
 * Fetch a remote JavaScript or JSON resource with multiple fallback methods.
 *
 * This function attempts to download the content of a remote URL using
 * Joomla's built-in HTTP client first. If that fails, it falls back to
 * cURL, and finally to PHP's `file_get_contents()` (if allowed).
 *
 * The function returns the raw response body on success, or `false` if
 * all methods fail. Error messages are also enqueued to the Joomla
 * application for administrator visibility.
 *
 */
public static function FetchContent($url, $timeout = 10)
{
	// Try Joomla HttpFactory
	if (class_exists('\Joomla\CMS\Http\HttpFactory')) {
		try {
			$http = HttpFactory::getHttp();
			$headers = [
				'User-Agent' => 'Joomla/' . JVERSION,
				'Accept'     => 'text/javascript, application/javascript, application/json, */*',
			];
			$response = $http->get($url, $headers, ['timeout' => $timeout]);
			if (is_object($response) && (int)$response->code === 200 && !empty($response->body)) {
				return (string) $response->body;
			}
		} catch (Exception $e) {
			Factory::getApplication()->enqueueMessage(
				Text::_('MOD_ADSIMPLE_ACM_ERROR_FETCH_SCRIPT'),
				'error'
			);
			Log::add( $e->getMessage(), Log::ERROR, 'adsimple-acm-fetch-failed' );
		}
	}

	// Try cURL fallback
	if (function_exists('curl_init')) {
		$ch = curl_init($url);
		curl_setopt_array($ch, [
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_TIMEOUT        => $timeout,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_MAXREDIRS      => 5,
			CURLOPT_USERAGENT      => 'Joomla/' . JVERSION,
			CURLOPT_HTTPHEADER     => ['Accept: text/javascript, application/javascript, application/json, */*'],
		]);
		$content  = curl_exec($ch);
		$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);
		if ($content !== false && $httpCode === 200) {
			return $content;
		}
	}

	// Last resort: file_get_contents
	if (ini_get('allow_url_fopen')) {
		$ctx = stream_context_create([
			'http' => [
				'timeout' => $timeout,
				'header'  => "User-Agent: Joomla/" . JVERSION . "\r\nAccept: */*\r\n"
			]
		]);
		$content = @file_get_contents($url, false, $ctx);
		if ($content !== false) {
			return $content;
		}
	}

	return false;
}


/**
 * Retrieve all module settings as an associative array.
 * This helper method collects the configuration parameters defined
 * in the module's XML manifest and returns them in a normalized array.
 */
public static function getSettings($params)
	{
		return [
			'acm_id' => $params->get('acm_id', ''),
			'cache_enabled' => $params->get('cache_enabled', ''),
			'cache_period_value' => $params->get('cache_period_value', ''),
			'cache_period_type' => $params->get('cache_period_type', ''),
			'cookie_list' => $params->get('cookie_list', ''),
		];
	}

}
